/***************************************************************************
 *   Copyright (C) 1998-2009 by David Bucciarelli (davibu@interfree.it)    *
 *                                                                         *
 *   This file is part of SmallLuxGPU.                                     *
 *                                                                         *
 *   SmallLuxGPU is free software; you can redistribute it and/or modify   *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *  SmallLuxGPU is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>. *
 *                                                                         *
 *   This project is based on PBRT ; see http://www.pbrt.org               *
 *   and Lux Renderer website : http://www.luxrender.net                   *
 ***************************************************************************/

#include <stdlib.h>

#include <stdexcept>
#include <sstream>

#include "scene.h"

using namespace std;

Scene::Scene(const string &fileName, const unsigned int screenWidth,
		const unsigned int screenHight) {
	cerr << "Reading scene: " << fileName << endl;

	ifstream file;
	file.exceptions(ifstream::eofbit | ifstream::failbit | ifstream::badbit);
	file.open(fileName.c_str(), ios::in);

	//--------------------------------------------------------------------------
	// Create Sampler
	//--------------------------------------------------------------------------

	sampler = new RandomSampler(screenWidth, screenHight);

	//--------------------------------------------------------------------------
	// Read light position and radius
	//--------------------------------------------------------------------------

	Spectrum lightGain;
	file >> lightGain.r;
	file >> lightGain.g;
	file >> lightGain.b;

	cerr << "Light gain: (" << lightGain.r << ", " << lightGain.g << ", " << lightGain.b << ")" << endl;

	//--------------------------------------------------------------------------
	// Read camera position and target
	//--------------------------------------------------------------------------

	Point o;
	file >> o.x;
	file >> o.y;
	file >> o.z;

	Point t;
	file >> t.x;
	file >> t.y;
	file >> t.z;

	cerr << "Camera postion: " << o << endl;
	cerr << "Camera target: " << t << endl;

	camera = new PerspectiveCamera(o, t, screenWidth, screenHight);

	//--------------------------------------------------------------------------
	// Read objects .ply file
	//--------------------------------------------------------------------------

	string plyFileName;
	file >> plyFileName;
	cerr << "PLY objects file name: " << plyFileName << endl;

	TriangleMesh objects(plyFileName);

	//--------------------------------------------------------------------------
	// Read lights .ply file
	//--------------------------------------------------------------------------

	file >> plyFileName;
	cerr << "PLY lights file name: " << plyFileName << endl;

	TriangleMesh lights(plyFileName);
	// Scale lights intensity
	for (unsigned int i = 0; i < lights.triangleCount; ++i)
		lights.colors[i] *= lightGain;

	//--------------------------------------------------------------------------
	// Join the ply objects
	//--------------------------------------------------------------------------

	meshLightOffset = objects.triangleCount;
	mesh = new TriangleMesh(objects, lights);

	cerr << "Vertex count: " << mesh->vertexCount << " (" << (mesh->vertexCount * sizeof(Point) / 1024) << "Kb)" << endl;
	cerr << "Triangle count: " << mesh->triangleCount << " (" << (mesh->triangleCount * sizeof(Triangle) / 1024) << "Kb)" << endl;

	//--------------------------------------------------------------------------
	// Create BVH
	//--------------------------------------------------------------------------

	const int treeType = 4; // Tree type to generate (2 = binary, 4 = quad, 8 = octree)
	const int costSamples = 0; // Samples to get for cost minimization
	const int isectCost = 80;
	const int travCost = 10;
	const float emptyBonus = 0.5f;
	bvh = new BVHAccel(mesh->triangleCount, mesh->triangles, mesh->vertices,
			treeType, costSamples, isectCost, travCost, emptyBonus);
}
