/***************************************************************************
 *   Copyright (C) 1998-2009 by David Bucciarelli (davibu@interfree.it)    *
 *                                                                         *
 *   This file is part of SmallLuxGPU.                                     *
 *                                                                         *
 *   SmallLuxGPU is free software; you can redistribute it and/or modify   *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *  SmallLuxGPU is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>. *
 *                                                                         *
 *   This project is based on PBRT ; see http://www.pbrt.org               *
 *   and Lux Renderer website : http://www.luxrender.net                   *
 ***************************************************************************/

#ifndef _SAMPLER_H
#define	_SAMPLER_H

#include "randomgen.h"

class Sample;

class Sampler {
public:
	virtual ~Sampler() { }

	virtual void Init(const unsigned width, const unsigned height) = 0;
	virtual unsigned int GetPass() = 0;

	virtual void GetNextSample(Sample *sample) = 0;
	virtual float GetLazyValue(Sample *sample) = 0;
};

class Sample {
public:
	Sample() { }

	void Init(Sampler *s,
		const unsigned x, const unsigned y,
		const float cx, const float cy,
		const unsigned p) {
		sampler = s;

		screenX = x;
		screenY = y;
		cameraX = cx;
		cameraY = cy;
		pass = p;
	}

	float GetLazyValue() {
		return sampler->GetLazyValue(this);
	}

	unsigned int screenX, screenY;
	float cameraX, cameraY;
	unsigned int pass;

private:
	Sampler *sampler;
};

class RandomSampler : public Sampler {
public:
	RandomSampler(const unsigned width, const unsigned height) {
		rndGen = new RandomGenerator();

		Init(width, height);
	}
	~RandomSampler() {
		delete rndGen;
	}

	void Init(const unsigned width, const unsigned height) {
		rndGen->init(1);
		screenWidth = width;
		screenHeight = height;
		currentSampleScreenX = 0;
		currentSampleScreenY = 0;
		pass = 0;
	}

	void GetNextSample(Sample *sample) {
		const float r1 = rndGen->floatValue() - .5f;
		const float r2 = rndGen->floatValue() - .5f;
		const float cx = (currentSampleScreenX + r1) / screenWidth - .5f;
		const float cy = (currentSampleScreenY + r2) / screenHeight - .5f;

		sample->Init(this,
				currentSampleScreenX, currentSampleScreenY,
				cx, cy,
				pass);

		currentSampleScreenX++;
		if (currentSampleScreenX == screenWidth) {
			currentSampleScreenX = 0;
			currentSampleScreenY++;

			if (currentSampleScreenY == screenHeight) {
				currentSampleScreenY = 0;
				pass++;
			}
		}
	}

	float GetLazyValue(Sample *sample) {
		return rndGen->floatValue();
	}

	unsigned int GetPass() { return pass; }

private:
	RandomGenerator *rndGen;
	unsigned int screenWidth, screenHeight;
	unsigned int currentSampleScreenX, currentSampleScreenY;
	unsigned int pass;
};

#endif	/* _FILM_H */
