/***************************************************************************
 *   Copyright (C) 1998-2009 by David Bucciarelli (davibu@interfree.it)    *
 *                                                                         *
 *   This file is part of SmallLuxGPU.                                     *
 *                                                                         *
 *   SmallLuxGPU is free software; you can redistribute it and/or modify   *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *  SmallLuxGPU is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>. *
 *                                                                         *
 *   This project is based on PBRT ; see http://www.pbrt.org               *
 *   and Lux Renderer website : http://www.luxrender.net                   *
 ***************************************************************************/

#ifndef _PATH_H
#define	_PATH_H

#define __CL_ENABLE_EXCEPTIONS
#define __NO_STD_VECTOR
#define __NO_STD_STRING

#if defined(__APPLE__) || defined(__MACOSX)
#include <OpenCL/cl.hpp>
#else
#include <CL/cl.hpp>
#endif

#include "point.h"
#include "light.h"
#include "scene.h"
#include "raybuffer.h"

class RenderingConfig;

class Path {
public:
	enum PathState {
		NEXT_VERTEX, DIRECT_LIGHT_SAMPLING
	};

	Path() { }

	void Init(Scene *scene, unsigned int maxPathDepth) {
		throughput = Spectrum(1.f, 1.f, 1.f);
		radiance = Spectrum(0.f, 0.f, 0.f);
		scene->sampler->GetNextSample(&sample);


		scene->camera->GenerateRay(&sample, &pathRay);
		state = NEXT_VERTEX;
		depth = 0;
		depthStop = min(maxPathDepth, sample.pass + 2);
	}

	void FillRayBuffer(RayBuffer *rayBuffer) {
		currentRayIndex = rayBuffer->currentFreeRay++;
		if (state == NEXT_VERTEX)
			rayBuffer->rays[currentRayIndex] = pathRay;
		else if (state == DIRECT_LIGHT_SAMPLING)
			rayBuffer->rays[currentRayIndex] = shadowRay;
	}

	void AdvancePath(Scene *scene, RayBuffer *rayBuffer, unsigned int maxPathDepth) {
		if (state == NEXT_VERTEX) {
			const RayHit *rayHit = &(rayBuffer->rayHits[currentRayIndex]);
			if (rayHit->index == 0xffffffffu) {
				// Hit nothing, terminate the ray
				scene->camera->film->SplatSample(&sample, radiance);
				// Restart the ray
				Init(scene, maxPathDepth);
				return;
			} else  {
				// Something was hit
				currentTriangleIndex = rayHit->index;

				// Check if it is a light source
				Normal N = scene->mesh->normals[currentTriangleIndex];
				float RdotN = Dot(pathRay.d, N);
				if (scene->IsLight(currentTriangleIndex)) {
					// Check if we are on the right side of the light source
					if ((depth == 0) && (RdotN < 0.f))
						radiance += scene->mesh->colors[currentTriangleIndex] * throughput;

					// Terminate the ray
					scene->camera->film->SplatSample(&sample, radiance);
					// Restart the ray
					Init(scene, maxPathDepth);
					return;
				}

				if (RdotN > 0.f) {
					// Flip the normal
					N = -N;
				} else
					RdotN = -RdotN;

				throughput *= RdotN * scene->mesh->colors[currentTriangleIndex];

				// Trace a shadow ray
				const Point hitPoint = pathRay(rayHit->t);

				// Select the light to sample
				currentLightIndex = scene->SampleLights(sample.GetLazyValue());
				const TriangleLight light(currentLightIndex, scene->mesh);

				// Select a point on the surface
				lightColor = light.Sample_L(hitPoint, N,
					sample.GetLazyValue(), sample.GetLazyValue(),
					&lightPdf, &shadowRay);
				state = DIRECT_LIGHT_SAMPLING;
				return;
			}
		} else if (state == DIRECT_LIGHT_SAMPLING) {
			const RayHit *rayHit = &(rayBuffer->rayHits[currentRayIndex]);

			Normal N = scene->mesh->normals[currentTriangleIndex];
			float RdotN = Dot(pathRay.d, N);
			if (RdotN > 0.f)
				N = -N;
			else
				RdotN = -RdotN;

			if ((rayHit->index == 0xffffffffu) && !lightColor.Black()) {
				// Nothing was hit, light is visible

				radiance += throughput * lightColor / lightPdf;
			}

			// Calculate next step
			depth++;

			// Check if I have to stop
			if (depth >= depthStop) {
				// Too depth, terminate the path
				scene->camera->film->SplatSample(&sample, radiance);
				// Restart the ray
				Init(scene, maxPathDepth);
				return;
			}

			// Calculate exit direction

			float r1 = 2.f * M_PI * sample.GetLazyValue();
			float r2 = sample.GetLazyValue();
			float r2s = sqrt(r2);
			const Vector w(N);

			Vector u;
			if (fabsf(N.x) > .1f) {
				const Vector a(0.f, 1.f, 0.f);
				u = Cross(a, w);
			} else {
				const Vector a(1.f, 0.f, 0.f);
				u = Cross(a, w);
			}
			u = Normalize(u);

			Vector v = Cross(w, u);

			Vector newDir = u * (cosf(r1) * r2s) + v * (sinf(r1) * r2s) + w * sqrtf(1.f - r2);
			newDir = Normalize(newDir);

			pathRay.o = shadowRay.o;
			pathRay.d = newDir;
			state = NEXT_VERTEX;

			if (depth > 1) {
				// Russian Rulette
				const float p = min(1.f, scene->mesh->colors[currentTriangleIndex].filter() * AbsDot(N, pathRay.d));
				if (p > sample.GetLazyValue())
					throughput /= p;
				else {
					// Terminate the ray
					scene->camera->film->SplatSample(&sample, radiance);
					// Restart the ray
					Init(scene, maxPathDepth);
					return;
				}
			}

			return;
		}
	}

private:
	Sample sample;

	Spectrum throughput, radiance;
	unsigned int depth, depthStop;

	Ray shadowRay;
	float lightPdf;
	Spectrum lightColor;

	Ray pathRay;
	unsigned int currentRayIndex;
	unsigned int currentTriangleIndex, currentLightIndex;
	PathState state;
};

enum AccelleratorType {
	NO_OPENCL_BVH, OPENCL_BRUTEFORCE, OPENCL_BVH
};

class PathIntegrator {
public:
	PathIntegrator(Scene *s, RayBuffer *rb, const bool useGPU);
	~PathIntegrator();

	void ReInit() {
		currentPath = 0;
		paths.clear();
	}

	size_t PathCount() const { return paths.size(); }

	void FillRayBuffer();
	void TraceRayBuffer(AccelleratorType type);
	void WaitForRayBuffer(AccelleratorType type);
	void AdvancePaths();

	void IncMaxPathDepth() {
		maxPathDepth++;
	}

	void DecMaxPathDepth() {
		maxPathDepth = max(1u, maxPathDepth - 1);
	}

	unsigned int GetMaxPathDepth() { return maxPathDepth; };

	double statsTotalRayTime;
	double statsTotalRayCount;

private:
	void SetUpOpenCL(const bool useGPU);
	cl::Kernel *SetUpKernel(const string &kernelFileName);
	string ReadSources(const string &fileName);

	unsigned int maxPathDepth;

	Scene *scene;
	RayBuffer *rayBuffer;
	unsigned int currentPath;
	vector<Path> paths;

	// OpenCL fields

	cl::Context *context;
	cl::vector<cl::Device> buildDevice;

	cl::Kernel *bruteforceKernel;
	size_t bruteforceWorkGroupSize;

	cl::Kernel *bvhKernel;
	size_t bvhWorkGroupSize;

	cl::CommandQueue *queue;

	cl::Buffer *raysBuff;
	cl::Buffer *hitsBuff;
	cl::Buffer *vertsBuff;
	cl::Buffer *trisBuff;
	cl::Buffer *bvhBuff;
};

#endif	/* _PATH_H */
