/***************************************************************************
 *   Copyright (C) 1998-2009 by David Bucciarelli (davibu@interfree.it)    *
 *                                                                         *
 *   This file is part of SmallLuxGPU.                                     *
 *                                                                         *
 *   SmallLuxGPU is free software; you can redistribute it and/or modify   *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *  SmallLuxGPU is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>. *
 *                                                                         *
 *   This project is based on PBRT ; see http://www.pbrt.org               *
 *   and Lux Renderer website : http://www.luxrender.net                   *
 ***************************************************************************/

#include <stdexcept>
#include <sstream>

#include "mesh.h"

// rply vertex callback
static int VertexCB(p_ply_argument argument) {
	long userIndex = 0;
	void *userData = NULL;
	ply_get_argument_user_data(argument, &userData, &userIndex);

	Point* p = *static_cast<Point **> (userData);

	long vertIndex;
	ply_get_argument_element(argument, NULL, &vertIndex);

	if (userIndex == 0)
		p[vertIndex].x =
			static_cast<float>(ply_get_argument_value(argument));
	else if (userIndex == 1)
		p[vertIndex].y =
			static_cast<float>(ply_get_argument_value(argument));
	else if (userIndex == 2)
		p[vertIndex].z =
			static_cast<float>(ply_get_argument_value(argument));

	return 1;
}

// rply color callback
static int ColorCB(p_ply_argument argument) {
	long userIndex = 0;
	void *userData = NULL;
	ply_get_argument_user_data(argument, &userData, &userIndex);

	Spectrum* p = *static_cast<Spectrum **> (userData);

	long vertIndex;
	ply_get_argument_element(argument, NULL, &vertIndex);

	if (userIndex == 0)
		p[vertIndex].r =
			static_cast<float>(ply_get_argument_value(argument) / 255.0);
	else if (userIndex == 1)
		p[vertIndex].g =
			static_cast<float>(ply_get_argument_value(argument) / 255.0);
	else if (userIndex == 2)
		p[vertIndex].b =
			static_cast<float>(ply_get_argument_value(argument) / 255.0);

	return 1;
}

// rply face callback
static int FaceCB(p_ply_argument argument) {
	void *userData = NULL;
	ply_get_argument_user_data(argument, &userData, NULL);

	Triangle *verts = *static_cast<Triangle **> (userData);

	long triIndex;
	ply_get_argument_element(argument, NULL, &triIndex);

	long length, valueIndex;
	ply_get_argument_property(argument, NULL, &length, &valueIndex);

	if (valueIndex >= 0 && valueIndex < 3) {
		verts[triIndex].v[valueIndex] =
				static_cast<unsigned int> (ply_get_argument_value(argument));
	}

	return 1;
}

TriangleMesh::TriangleMesh(const string &fileName) {
	p_ply plyfile = ply_open(fileName.c_str(), NULL);
	if (!plyfile) {
		stringstream ss;
		ss << "Unable to read PLY mesh file '" << fileName << "'";
		throw runtime_error(ss.str());
	}

	if (!ply_read_header(plyfile)) {
		stringstream ss;
		ss << "Unable to read PLY header from '" << fileName << "'";
		throw runtime_error(ss.str());
	}

	Point *p;
	long plyNbVerts = ply_set_read_cb(plyfile, "vertex", "x", VertexCB, &p, 0);
	ply_set_read_cb(plyfile, "vertex", "y", VertexCB, &p, 1);
	ply_set_read_cb(plyfile, "vertex", "z", VertexCB, &p, 2);
	if (plyNbVerts <= 0) {
		stringstream ss;
		ss << "No vertices found in '" << fileName << "'";
		throw runtime_error(ss.str());
	}

	Triangle *vertexIndex;
	long plyNbTris = ply_set_read_cb(plyfile, "face", "vertex_indices", FaceCB, &vertexIndex, 0);
	if (plyNbTris <= 0) {
		stringstream ss;
		ss << "No triangles found in '" << fileName << "'";
		throw runtime_error(ss.str());
	}

	Spectrum *c;
	long plyNbColors = ply_set_read_cb(plyfile, "vertex", "red", ColorCB, &c, 0);
	ply_set_read_cb(plyfile, "vertex", "green", ColorCB, &c, 1);
	ply_set_read_cb(plyfile, "vertex", "blue", ColorCB, &c, 2);

	if ((plyNbColors > 0) && (plyNbColors != plyNbVerts)) {
		stringstream ss;
		ss << "Wrong count of colors in '" << fileName << "'";
		throw runtime_error(ss.str());
	}

	p = new Point[plyNbVerts];
	vertexIndex = new Triangle[plyNbTris];
	c = new Spectrum[plyNbColors];
	if (plyNbColors <= 0) {
		for (unsigned int i = 0; i < plyNbColors; ++i) {
			c[i].r = 0.7f;
			c[i].g = 0.7f;
			c[i].b = 0.7f;
		}
	}

	if (!ply_read(plyfile)) {
		stringstream ss;
		ss << "Unable to parse PLY file '" << fileName << "'";

		delete[] p;
		delete[] vertexIndex;
		delete[] c;

		throw runtime_error(ss.str());
	}

	ply_close(plyfile);

	vertexCount = plyNbVerts;
	vertices = p;
	triangleCount = plyNbTris;
	triangles = static_cast<Triangle *>(vertexIndex);

	// Calculate triangle color
	colors = new Spectrum[triangleCount];
	for (unsigned int i = 0; i < triangleCount; ++i) {
		const Spectrum &c0 = c[triangles[i].v[0]];
		const Spectrum &c1 = c[triangles[i].v[1]];
		const Spectrum &c2 = c[triangles[i].v[2]];
		colors[i] =  0.75f * (c0 + c1 +c2) / 3.f;
	}
	delete[] c;

	// Calculate normals
	normals = new Normal[triangleCount];
	for (unsigned int i = 0; i < triangleCount; ++i) {
		const Vector e1 = vertices[triangles[i].v[1]] - vertices[triangles[i].v[0]];
		const Vector e2 = vertices[triangles[i].v[2]] - vertices[triangles[i].v[0]];
		normals[i] = Normal(Normalize(Cross(e1, e2)));
	}
}

TriangleMesh::~TriangleMesh() {
	delete vertices;
	delete triangles;
	delete colors;
	delete normals;
}

TriangleMesh::TriangleMesh(const TriangleMesh &obj0, const TriangleMesh &obj1) {
	vertexCount = obj0.vertexCount + obj1.vertexCount;
	triangleCount = obj0.triangleCount + obj1.triangleCount;

	vertices = new Point[vertexCount];
	triangles = new Triangle[triangleCount];
	colors = new Spectrum[triangleCount];
	normals = new Normal[triangleCount];

	for (unsigned int i = 0; i < obj0.vertexCount; ++i)
		vertices[i] = obj0.vertices[i];
	for (unsigned int i = 0; i < obj1.vertexCount; ++i)
		vertices[obj0.vertexCount + i] = obj1.vertices[i];

	for (unsigned int i = 0; i < obj0.triangleCount; ++i) {
		triangles[i] = obj0.triangles[i];
		colors[i] = obj0.colors[i];
		normals[i] = obj0.normals[i];
	}
	for (unsigned int i = 0; i < obj1.triangleCount; ++i) {
		triangles[obj0.triangleCount + i].v[0] = obj1.triangles[i].v[0] + obj0.vertexCount;
		triangles[obj0.triangleCount + i].v[1] = obj1.triangles[i].v[1] + obj0.vertexCount;
		triangles[obj0.triangleCount + i].v[2] = obj1.triangles[i].v[2] + obj0.vertexCount;
		colors[obj0.triangleCount + i] = obj1.colors[i];
		normals[obj0.triangleCount + i] = obj1.normals[i];
	}
}