/***************************************************************************
 *   Copyright (C) 1998-2009 by David Bucciarelli (davibu@interfree.it)    *
 *                                                                         *
 *   This file is part of SmallLuxGPU.                                     *
 *                                                                         *
 *   SmallLuxGPU is free software; you can redistribute it and/or modify   *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *  SmallLuxGPU is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>. *
 *                                                                         *
 *   This project is based on PBRT ; see http://www.pbrt.org               *
 *   and Lux Renderer website : http://www.luxrender.net                   *
 ***************************************************************************/

#ifndef _FILM_H
#define	_FILM_H

#include <cstddef>
#include <cmath>

#include "smalllux.h"
#include "vector.h"
#include "spectrum.h"
#include "sampler.h"

#define GAMMA_TABLE_SIZE 1024

class Film {
public:
	Film(const unsigned int w, unsigned int h) {
		pixelsRadiance = NULL;
		pixelSampleCount = NULL;
		pixels = NULL;

		InitGammaTable();

		Init(w, h);
	}
	~Film() {
		if (pixelsRadiance)
			delete pixelsRadiance;
		if (pixelSampleCount)
			delete pixelSampleCount;
		if (pixels)
			delete pixels;
	}

	void Init(const unsigned int w, unsigned int h) {
		if (pixelsRadiance)
			delete pixelsRadiance;
		if (pixelSampleCount)
			delete pixelSampleCount;
		if (pixels)
			delete pixels;

		width = w;
		height = h;
		//cerr << "Film size " << width << "x" << height << endl;

		pixelCount = width * height;
		statsTotalSampleCount = 0.0;

		pixelsRadiance = new Spectrum[pixelCount];
		pixels = new float[pixelCount * 3];
		pixelSampleCount = new unsigned int[pixelCount];

		for (unsigned int i = 0, j = 0; i < pixelCount; ++i) {
			pixelsRadiance[i] = 0.f;
			pixelSampleCount[i] = 0;
			pixels[j++] = 0.f;
			pixels[j++] = 0.f;
			pixels[j++] = 0.f;
		}

		statsStartSampleTime = WallClockTime();
	}

	void Reset() {
		for (unsigned int i = 0; i < pixelCount; ++i)
			pixelSampleCount[i] = 0;
	}

	void UpdateScreenBuffer() {
		unsigned int count = width * height;
		for (unsigned int i = 0, j = 0; i < count; ++i) {
			pixels[j++] = Radiance2PixelFloat(pixelsRadiance[i].r);
			pixels[j++] = Radiance2PixelFloat(pixelsRadiance[i].g);
			pixels[j++] = Radiance2PixelFloat(pixelsRadiance[i].b);
		}
	}

	const float *GetScreenBuffer() const {
		return pixels;
	}

	void SplatSample(const Sample *sample, const Spectrum &radiance) {
		const unsigned int offset = sample->screenX + sample->screenY * width;
		const float k1 = pixelSampleCount[offset] / (pixelSampleCount[offset] + 1.f);
		const float k2 = 1.f - k1;

		pixelsRadiance[offset] = pixelsRadiance[offset] * k1 + k2 * radiance;
		pixelSampleCount[offset]++;

		statsTotalSampleCount += 1.0;
	}

	unsigned int GetWidth() { return width; }
	unsigned int GetHeight() { return height; }
	double GetTotalSampleCount() { return statsTotalSampleCount; }
	double GetTotalTime() {
		return WallClockTime() - statsStartSampleTime;
	}
	double GetAvgSampleSec() {
		return statsTotalSampleCount / (WallClockTime() - statsStartSampleTime);
	}

	void SavePPM(const string &fileName) {
		ofstream file;
		file.exceptions(ifstream::eofbit | ifstream::failbit | ifstream::badbit);
		file.open(fileName.c_str(), ios::out);

		file << "P3\n" << width << " " << height << "\n255\n";

		// Update pixels
		GetScreenBuffer();

		for (unsigned int y = 0; y < height; ++y) {
			for (unsigned int x = 0; x < width; ++x) {
				const int offset = 3 * (x + (height - y - 1) * width);
				const int r = (int)(pixels[offset] * 255.f + .5f);
				const int g = (int)(pixels[offset + 1] * 255.f + .5f);
				const int b = (int)(pixels[offset + 2] * 255.f + .5f);

				file << r << " " << g << " " << b << " ";
			}
		}

		file.close();
	}

private:
	float gammaTable[GAMMA_TABLE_SIZE];

	void InitGammaTable() {
		float x = 0.f;
		const float dx = 1.f / GAMMA_TABLE_SIZE;
		for (int i = 0; i < GAMMA_TABLE_SIZE; ++i, x += dx)
			gammaTable[i] = powf(Clamp(x, 0.f, 1.f), 1.f / 2.2f);
	}

	float Radiance2PixelFloat(const float x) const {
		// Very slow !
		//return powf(Clamp(x, 0.f, 1.f), 1.f / 2.2f);

		const unsigned int index = min<unsigned int>(
			Floor2UInt(GAMMA_TABLE_SIZE * Clamp(x, 0.f, 1.f)),
				GAMMA_TABLE_SIZE - 1);
		return gammaTable[index];
	}

	unsigned int width, height;
	unsigned int pixelCount;

	double statsTotalSampleCount;
	double statsStartSampleTime;

	Spectrum *pixelsRadiance;
	unsigned int *pixelSampleCount;
	float *pixels;
};

#endif	/* _FILM_H */
