/***************************************************************************
 *   Copyright (C) 1998-2009 by David Bucciarelli (davibu@interfree.it)    *
 *                                                                         *
 *   This file is part of SmallLuxGPU.                                     *
 *                                                                         *
 *   SmallLuxGPU is free software; you can redistribute it and/or modify   *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *  SmallLuxGPU is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>. *
 *                                                                         *
 *   This project is based on PBRT ; see http://www.pbrt.org               *
 *   and Lux Renderer website : http://www.luxrender.net                   *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#ifdef WIN32
#define _USE_MATH_DEFINES
#endif
#include <math.h>

#include "displayfunc.h"
#include "film.h"

RenderingConfig *config;

static int printHelp = 1;

static double lastOverheadTime = 0.0;

static void UpdateRendering() {
	const double t1 = WallClockTime();
	const float overheadTime = t1 - lastOverheadTime;

	// Fill the list of ray to trace
	config->pathIntegrator->FillRayBuffer();

	// Trace rays
	const double statsTotalRayCount = config->pathIntegrator->statsTotalRayCount;
	const double t2 = WallClockTime();

	config->pathIntegrator->TraceRayBuffer(config->acceleratorType);

	const double t3 = WallClockTime();
	const double rayCount = config->pathIntegrator->statsTotalRayCount - statsTotalRayCount;

	// Collect results
	config->pathIntegrator->AdvancePaths();
	const double t4 = WallClockTime();
	lastOverheadTime = t3;

	const double cpuTime = (t2 - t1) + (t4 - t3);
	const double rayTime = t3 - t2;
	const double totalTime = cpuTime + rayTime + overheadTime;

	sprintf(config->captionBuffer1, "[Samples %4d][Refresh %3dms (CPU %3dms + %s %3dms + Overhead %3dms)]",
			config->scene->sampler->GetPass(),
			int(1000.0 * totalTime), int(1000.0 * cpuTime),
			(config->acceleratorType == NO_OPENCL_BVH) ? "CPU" : "GPU",
			int(1000.0 * rayTime), int(1000.0 * overheadTime));

	const double raysSec = rayCount / rayTime;
	const double samplesSecTot = config->scene->camera->film->GetAvgSampleSec();
	const double raysSecTot = config->pathIntegrator->statsTotalRayCount / config->pathIntegrator->statsTotalRayTime;
	sprintf(config->captionBuffer0, "[Avg. samples/sec %6.1fK][Rays/sec %6.1fK (Avg. %6.1fK) on %.1fK tris]",
			samplesSecTot / 1000.0, raysSec / 1000.0, raysSecTot / 1000.0, config->scene->mesh->triangleCount / 1000.0);
}

static void ReInit(const bool reallocBuffers, const unsigned int w = 0, unsigned int h = 0) {
	// Check if I have to reallocate buffers
	if (reallocBuffers)
		config->scene->camera->film->Init(w, h);
	else
		config->scene->camera->film->Reset();
	config->scene->camera->Update();

	config->scene->sampler->Init(config->scene->camera->film->GetWidth(), config->scene->camera->film->GetHeight());
	config->rayBuffer->currentFreeRay = 0;
	config->pathIntegrator->ReInit();
}

static void PrintString(void *font, const char *string) {
	int len, i;

	len = (int)strlen(string);
	for (i = 0; i < len; i++)
		glutBitmapCharacter(font, string[i]);
}

static void PrintHelpAndSettings() {
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glColor4f(0.f, 0.f, 0.f, 0.5f);
	glRecti(40, 80, 600, 440);
	glDisable(GL_BLEND);

	glColor3f(1.f, 1.f, 1.f);
	glRasterPos2i(300, 420);
	PrintString(GLUT_BITMAP_9_BY_15, "Help & Settings");

	// Help
	glRasterPos2i(60, 390);
	PrintString(GLUT_BITMAP_9_BY_15, "h - toggle Help");
	glRasterPos2i(60, 370);
	PrintString(GLUT_BITMAP_9_BY_15, "arrow Keys - rotate camera");
	glRasterPos2i(60, 350);
	PrintString(GLUT_BITMAP_9_BY_15, "a, s, d, w - move camera");
	glRasterPos2i(60, 330);
	PrintString(GLUT_BITMAP_9_BY_15, "p - save image.ppm");
	glRasterPos2i(60, 310);
	PrintString(GLUT_BITMAP_9_BY_15, "1 - select no-OpenCL BVH accelerator (very slow !)");
	glRasterPos2i(60, 290);
	PrintString(GLUT_BITMAP_9_BY_15, "2 - select OpenCL BruteForce accelerator (very slow !)");
	glRasterPos2i(60, 270);
	PrintString(GLUT_BITMAP_9_BY_15, "3 - select OpenCL BVH accelerator");
	glRasterPos2i(60, 250);
	PrintString(GLUT_BITMAP_9_BY_15, "n, m - decrease/increase the minimum screen refresh time");
	glRasterPos2i(60, 230);
	PrintString(GLUT_BITMAP_9_BY_15, "v, b - decrease/increase the max. path depth");
	glRasterPos2i(60, 210);
	PrintString(GLUT_BITMAP_9_BY_15, "x, c - decrease/increase the field of view");

	// Settings
	char buf[512];
	glRasterPos2i(45, 120);
	PrintString(GLUT_BITMAP_8_BY_13, "Settings:");
	glRasterPos2i(45, 105);
	sprintf(buf, "[Rendering time: %dsecs][Max. path depth: %d][FOV: %.1f]",
			int(config->scene->camera->film->GetTotalTime()),
			config->pathIntegrator->GetMaxPathDepth(),
			config->scene->camera->fieldOfView);
	PrintString(GLUT_BITMAP_8_BY_13, buf);
	glRasterPos2i(45, 90);
	sprintf(buf, "[Accel. type: %s][Active paths: %lu][Min. refresh: %dms]",
			(config->acceleratorType == NO_OPENCL_BVH) ?
			"OpenCL disabled" : ((config->acceleratorType == OPENCL_BRUTEFORCE) ?
			"OpenCL BruteForce" : ((config->acceleratorType == OPENCL_BVH) ?
			"OpenCL BVH" : "Unknown")), config->pathIntegrator->PathCount(),
			int(1000.0 * config->minScreenRefreshInterval));
	PrintString(GLUT_BITMAP_8_BY_13, buf);
}

static void PrintCaptions() {
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glColor4f(0.f, 0.f, 0.f, 0.8f);
	glRecti(0, config->scene->camera->film->GetHeight() - 15,
			config->scene->camera->film->GetWidth() - 1, config->scene->camera->film->GetHeight() - 1);
	glRecti(0, 0, config->scene->camera->film->GetWidth() - 1, 35);
	glDisable(GL_BLEND);

	// Caption line 0
	glColor3f(1.f, 1.f, 1.f);
	glRasterPos2i(4, 5);
	PrintString(GLUT_BITMAP_8_BY_13, config->captionBuffer0);
	glRasterPos2i(4, 20);
	PrintString(GLUT_BITMAP_8_BY_13, config->captionBuffer1);
	// Title
	glRasterPos2i(4, config->scene->camera->film->GetHeight() - 10);
	PrintString(GLUT_BITMAP_8_BY_13, "SmallLuxGPU V1.0 (Written by David Bucciarelli)");
}

static double lastDisplayTime = 0;
void displayFunc(void) {
	const double t = WallClockTime();
	if (t - lastDisplayTime < config->minScreenRefreshInterval) {
		// Update the display max once every minScreenRefreshInterval
		return;
	} else
		lastDisplayTime = t;

	config->scene->camera->film->UpdateScreenBuffer();
	const float *pixels = config->scene->camera->film->GetScreenBuffer();

	glRasterPos2i(0, 0);
	glDrawPixels(config->scene->camera->film->GetWidth(), config->scene->camera->film->GetHeight(), GL_RGB, GL_FLOAT, pixels);

	PrintCaptions();

	if (printHelp) {
		glPushMatrix();
		glLoadIdentity();
		glOrtho(-0.5, 639.5, -0.5, 479.5, -1.0, 1.0);

		PrintHelpAndSettings();

		glPopMatrix();
	}

	glutSwapBuffers();
}

void reshapeFunc(int newWidth, int newHeight) {
	glViewport(0, 0, newWidth, newHeight);
	glLoadIdentity();
	glOrtho(0.f, newWidth - 1.0f, 0.f, newHeight - 1.0f, -1.f, 1.f);

	ReInit(true, newWidth, newHeight);

	glutPostRedisplay();
}

#define MOVE_STEP 0.5f
#define ROTATE_STEP 4.f
void keyFunc(unsigned char key, int x, int y) {
	switch (key) {
		case 'p': {
			config->scene->camera->film->SavePPM("image.ppm");
			break;
		}
		case 27: // Escape key
			cerr << "Done." << endl;
			delete config;
			exit(0);
			break;
		case ' ': // Restart rendering
			ReInit(true, config->scene->camera->film->GetWidth(), config->scene->camera->film->GetHeight());
			break;
		case 'a': {
			config->scene->camera->TranslateLeft(MOVE_STEP);
			ReInit(false);
			break;
		}
		case 'd': {
			config->scene->camera->TranslateRight(MOVE_STEP);
			ReInit(false);
			break;
		}
		case 'w': {
			config->scene->camera->TranslateForward(MOVE_STEP);
			ReInit(false);
			break;
		}
		case 's': {
			config->scene->camera->TranslateBackward(MOVE_STEP);
			ReInit(false);
			break;
		}
		case 'r':
			config->scene->camera->Translate(Vector(0.f, 0.f, MOVE_STEP));
			ReInit(false);
			break;
		case 'f':
			config->scene->camera->Translate(Vector(0.f, 0.f, -MOVE_STEP));
			ReInit(false);
			break;
		case 'h':
			printHelp = (!printHelp);
			break;
		case '1':
			config->acceleratorType = NO_OPENCL_BVH;
			break;
		case '2':
			config->acceleratorType = OPENCL_BRUTEFORCE;
			break;
		case '3':
			config->acceleratorType = OPENCL_BVH;
			break;
		case 'x':
			config->scene->camera->fieldOfView = max(15.f,
					config->scene->camera->fieldOfView - 5.f);
			ReInit(false);
			break;
		case 'c':
			config->scene->camera->fieldOfView = min(180.f,
					config->scene->camera->fieldOfView + 5.f);
			ReInit(false);
			break;
		case 'v':
			config->pathIntegrator->DecMaxPathDepth();
			ReInit(false);
			break;
		case 'b':
			config->pathIntegrator->IncMaxPathDepth();
			ReInit(false);
			break;
		case 'n':
			config->minScreenRefreshInterval = max(0.05, config->minScreenRefreshInterval - 0.05);
			break;
		case 'm':
			config->minScreenRefreshInterval += 0.05;
			break;
		default:
			break;
	}

	glutPostRedisplay();
}

void specialFunc(int key, int x, int y) {
	switch (key) {
		case GLUT_KEY_UP:
			config->scene->camera->RotateUp(ROTATE_STEP);
			break;
		case GLUT_KEY_DOWN:
			config->scene->camera->RotateDown(ROTATE_STEP);
			break;
		case GLUT_KEY_LEFT:
			config->scene->camera->RotateLeft(ROTATE_STEP);
			break;
		case GLUT_KEY_RIGHT:
			config->scene->camera->RotateRight(ROTATE_STEP);
			break;
		default:
			break;
	}

	ReInit(false);
	glutPostRedisplay();
}

/*static int mouseButton0 = 0;
static int mouseButton2 = 0;
static int mouseGrabLastX = 0;
static int mouseGrabLastY = 0;*/

void mouseFunc(int button, int state, int x, int y) {
}

void motionFunc(int x, int y) {
}

void idleFunc(void) {
	UpdateRendering();

	glutPostRedisplay();
}

void InitGlut(int argc, char *argv[]) {
	glutInitWindowSize(config->scene->camera->film->GetWidth(), config->scene->camera->film->GetHeight());
	glutInitWindowPosition(0, 0);
	glutInitDisplayMode(GLUT_RGB | GLUT_DOUBLE);
	glutInit(&argc, argv);

	glutCreateWindow("SmallLuxGPU V1.0 (Written by David Bucciarelli)");

	glutReshapeFunc(reshapeFunc);
	glutKeyboardFunc(keyFunc);
	glutSpecialFunc(specialFunc);
	glutDisplayFunc(displayFunc);
	glutMouseFunc(mouseFunc);
	glutMotionFunc(motionFunc);
	glutIdleFunc(idleFunc);

	glMatrixMode(GL_PROJECTION);

	lastOverheadTime = WallClockTime();
}
