/***************************************************************************
 *   Copyright (C) 1998-2009 by David Bucciarelli (davibu@interfree.it)    *
 *                                                                         *
 *   This file is part of SmallLuxGPU.                                     *
 *                                                                         *
 *   SmallLuxGPU is free software; you can redistribute it and/or modify   *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *  SmallLuxGPU is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>. *
 *                                                                         *
 *   This project is based on PBRT ; see http://www.pbrt.org               *
 *   and Lux Renderer website : http://www.luxrender.net                   *
 ***************************************************************************/

typedef struct {
	float x, y, z;
} Point;

typedef struct {
	float x, y, z;
} Vector;

typedef struct {
	Point o;
	Vector d;
	float mint, maxt;
} Ray;

typedef struct {
	unsigned int index;
	float t;
} RayHit;

typedef struct {
	unsigned int v[3];
} Triangle;

static void TriangleIntersect(
		const float4 rayOrig,
		const float4 rayDir,
		const float minT,
		float *maxT,
		unsigned int *hitIndex,
		const unsigned int currentIndex,
		__constant Point *verts,
		__constant Triangle *tris) {

	// Load triangle vertices
	Point p0 = verts[tris[currentIndex].v[0]];
	Point p1 = verts[tris[currentIndex].v[1]];
	Point p2 = verts[tris[currentIndex].v[2]];

	float4 v0 = (float4) (p0.x, p0.y, p0.z, 0.f);
	float4 v1 = (float4) (p1.x, p1.y, p1.z, 0.f);
	float4 v2 = (float4) (p2.x, p2.y, p2.z, 0.f);

	// Calculate intersection
	float4 e1 = v1 - v0;
	float4 e2 = v2 - v0;
	float4 s1 = cross(rayDir, e2);

	const float divisor = dot(s1, e1);
	if (divisor == 0.f)
		return;

	const float invDivisor = 1.f / divisor;

	// Compute first barycentric coordinate
	const float4 d = rayOrig - v0;
	const float b1 = dot(d, s1) * invDivisor;
	if (b1 < 0.f)
		return;

	// Compute second barycentric coordinate
	const float4 s2 = cross(d, e1);
	const float b2 = dot(rayDir, s2) * invDivisor;
	if (b2 < 0.f)
		return;

	const float b0 = 1.f - b1 - b2;
	if (b0 < 0.f)
		return;

	// Compute _t_ to intersection point
	const float t = dot(e2, s2) * invDivisor;
	if (t < minT || t > *maxT)
		return;

	*maxT = t;
	*hitIndex = currentIndex;
}

__kernel void Intersect(
		__global Ray *rays,
		__global RayHit *rayHits,
		__constant Point *verts,
		__constant Triangle *tris,
		const unsigned int triangleCount) {
	// Select the ray to check
	const int gid = get_global_id(0);

	float4 rayOrig = (float4) (rays[gid].o.x, rays[gid].o.y, rays[gid].o.z, 0.f);
	float4 rayDir = (float4) (rays[gid].d.x, rays[gid].d.y, rays[gid].d.z, 0.f);

	// Check all triangles
	float minT = rays[gid].mint;
	float maxT = rays[gid].maxt;
	unsigned int hitIndex = 0xffffffffu;
	unsigned int i;
	for (i = 0; i < triangleCount; ++i)
		TriangleIntersect(rayOrig, rayDir, minT, &maxT, &hitIndex, i, verts, tris);

	// Write result
	rayHits[gid].index = hitIndex;
	rayHits[gid].t = maxT;
}
